// Stand mount for the piezo force hammer providing multiple degrees of adjustment.
// Mark Rau, 2020
// mrau@ccrma.stanford.edu
//
// The stand base is edited from a project created by Zuzzuk (https://www.thingiverse.com/thing:2548892)
//
//
// Use:
// 1. A 14mm nut fits in the stand mount. Adjust the nut_size and nut_height for a different sized nut. A nut with 3/8" inner threads works well with a standard Euro microphone stand adapter. 
// 2. The parts can be connected together using bolts and standard nuts or wingnuts.
// 3. The interiot of the hammer attachment should come in contact with only the center cylinder of the bearing, allowing the outside cylinder and hammer to swing freely.
//
//
// Licensed under the Creative Commons - Attribution license.

// Which part would you like to see?
part = "all"; 

/* [Basic] */

// Nut parameters in a mount

// Size of a nut between two flat edges
nut_size = 14.2;

nut_height = 5.6;

// Microphone holder parameters
holder_length = 30;

// Front inner diameter must be bigger or equal than rear one
front_inner_diameter = 12.8;
rear_inner_diameter = 12.8;

// Holder wall thickness
wall_thickness = 1.7;

/* [Advanced] */

// Knurled mount parameters
mount_base_height = 12;
mount_base_diameter = 28;

// Knurl's depth
knurl_depth = 2;
knurl_count = 10;

// Joint parameters
joint_outer_diameter = 16;

// Distance between two outer surfaces of the joint
joint_total_thickness = 12;
joint_height = 18;

// Gap width for middle joint arm
joint_slot_width = 4;

// Hole for the joint axle
joint_hole_diameter = 7;


/* [Print options] */

// printing parameters

// Expansion of detail as a result of extrusion
print_expansion = 0.1;

// Gap between details when printed together
gap_between_parts = 2;

/* [Hidden] */

// construction hidden constants

// technical overlap between joined or deducted details to ensure volumes are combined in one and deducted without thin planes at the edges
overlap=0.1;
pi=3.14;

// all parts together

print_part();

module print_part() {
    if ( (part == "all") || (part == "mount") ) {
        base_part();
    }

    if ( (part == "all") || (part == "holder") ) {
        offset=(mount_base_diameter/2)*cos(asin(((front_inner_diameter/2+wall_thickness)*sin(45))/(mount_base_diameter/2)))+(front_inner_diameter/2+wall_thickness)*(1+cos(45))+wall_thickness;
        translate( [offset+gap_between_parts,0,0] )
        rotate( [0,0,-90] )
        holder_part();
    }
    
    
    translate([00,50,49])
    rotate([90,0,0])
    holder_part2();
}

// PART 1: Stand Mount

module base_part() {
    union() {
        difference() {
            base();
            nut ();
            knurls();
        };

        translate( [0,0,mount_base_height] )
        difference(){
            stand_mount_joint();
            translate([0,-4.5,10])
            rotate([90,0,0])
            nut2();
        }
    }
}

module base () {
    cylinder( $fn=mount_base_diameter*pi*3, mount_base_height, d=mount_base_diameter );
}    

module knurls () {
    count = knurl_count;
    height = mount_base_height;
    radius = knurl_depth;
    surface_diameter = mount_base_diameter;
    
    angle=360/count;
    for( number = [1 : count] ) {
        rotate( [0,0,angle*number] )
        translate( [surface_diameter/2,0,-overlap] )
        cylinder( $fn=24, height+2*overlap, r=radius );
    };
}

module nut () {
    translate( [0,0,-overlap] )
    cylinder( $fn=6, nut_height+print_expansion+overlap, d=((nut_size+2*print_expansion)/cos(30)) );
}

module stand_mount_joint () {
    width = joint_outer_diameter;
    depth = joint_total_thickness;
    height = joint_height;
    hole_diameter = joint_hole_diameter+2*print_expansion;
	
	slot_width = joint_slot_width+2*print_expansion + 0.1; // Add some offset here because it was tight

	union() {
		// joint 
		difference() {
			// joint full body
			union() {
				// cubical part of the joint is radius less than full height
				// radius is half width of the joint
				translate( [-width/2, -depth/2, -overlap] )
				cube( [width, depth, height-width/2+overlap] );
				
				// radial part of the joint
				translate( [0,0,height-width/2] )
				rotate( [90,0,0] ) 
				cylinder( $fn=width*pi*3, depth, d=width, center=true );
			}
			
			// slot
			translate( [-width/2-overlap, -slot_width/2, -overlap*2] )
			cube( [width+overlap*2, slot_width, height+overlap*3] );
			
			// mount hole
			translate( [0,0,height-width/2] )
			rotate( [90,0,0] ) 
			cylinder( $fn=hole_diameter*pi*3, depth+overlap*2, d=hole_diameter, center=true );
		}
		
	}
    
}











// PART 2: Middle Attachment

// this is the angle for whole detail to rotate to make holder semi-circle perpendicular to XY plane
ratio = front_inner_diameter/rear_inner_diameter;
tilt_angle = atan(((rear_inner_diameter/2+wall_thickness)*ratio-(rear_inner_diameter/2+wall_thickness))/holder_length);

module holder_part() {
    // make wide side of the holder laying flat at XY plane
    translate ( [0,20,8] )
    rotate( [-90+tilt_angle,0,0] )
    union() {
        //holder();

        holder_joint();
        rotate([0,180,0])
        translate( [0,0,5] )
        
        difference(){
            stand_mount_joint();
            translate([0,-4.5,10])
            rotate([90,0,0])
            nut2();
        }
        
        translate([0,-8,2.5])
        rotate([0,135,0])
        prism(16, 11, 11);
        
        
    }
}

module holder() {
	length = holder_length;
	inner_radius = [rear_inner_diameter/2, front_inner_diameter/2];
	thickness = wall_thickness;
	
	outer_radius = [inner_radius[0]+thickness, (inner_radius[0]+thickness)*ratio];
	
    // rotate holder to make it horizontal before attaching to the joint
	rotate( [-tilt_angle,0,0] )
    translate( [0,0,-outer_radius[0]*(1+ratio)/2] )
    rotate( [-90,0,0] )
	rotate( [0,0,45] )
	linear_extrude( height=length, slices=ceil(length), scale=ratio, center=true )
	union() {
		difference() {
			circle( $fn=2*outer_radius[0]*pi*3, r=outer_radius[0] );
			circle( $fn=2*inner_radius[0]*pi*3, r=inner_radius[0] );
			square( outer_radius[0]+overlap );
		}
		
		translate( [inner_radius[0]+thickness/2, 0] )
		circle( $fn=12, d=thickness );

		translate( [0, inner_radius[0]+thickness/2] )
		circle( $fn=12, d=thickness );
	}
}


module holder_joint () {
    width = joint_outer_diameter;
    depth = joint_slot_width-print_expansion;
    height = joint_height;
    hole_diameter = joint_hole_diameter+2*print_expansion;

	holder_min_radius = rear_inner_diameter/2+wall_thickness;
	insert_angle = asin((depth/2)/holder_min_radius);
	insert_length = holder_min_radius*(1-cos(insert_angle))+overlap;

	translate( [0, 0, -insert_length/2] )
	cube( [depth, width, insert_length], center=true );
	
	union() {
		// joint without cogs
		difference() {
			// joint full body
			union() {
				// cubical part of the joint
				translate( [-depth/2, -width/2, -overlap] )
				cube( [depth, width, height-width/2+overlap] );
				
				// radial part of the joint
				translate( [0,0,height-width/2] )
				rotate( [0,90,0] ) 
				cylinder( $fn=width*pi*3, depth, d=width, center=true );
			}
			
			// mount hole
			translate( [0,0,height-width/2] )
			rotate( [0,90,0] ) 
			cylinder( $fn=hole_diameter*pi*3, depth+overlap*2, d=hole_diameter, center=true );

		}
	}
}

module nut2 () {
            translate( [0,0,0] )
            cylinder( $fn=6, 3, d=12.6 );
        }










// PART 3: Hammer Attachment

// this is the angle for whole detail to rotate to make holder semi-circle perpendicular to XY plane
ratio = front_inner_diameter/rear_inner_diameter;
tilt_angle = atan(((rear_inner_diameter/2+wall_thickness)*ratio-(rear_inner_diameter/2+wall_thickness))/holder_length);

module holder_part2() {
    // make wide side of the holder laying flat at XY plane
    translate ( [0,-40,8] )
    rotate( [-90+tilt_angle,0,90] )
    union() {
        //holder();
        
        // single joint
        holder_joint();

        //triangle between
        translate([0,-8,2.5])
        rotate([0,135,0])
        prism(16, 12.8, 12.8);
        

        
        
        
        //mount variables
        hammCylOuterD = 29;
        
        // mount for hammer 
        difference(){
            union(){
        rotate([90,0,90])
        translate([0,-23,-8.75])
        cylinder(9.5+4*2,hammCylOuterD/2,hammCylOuterD/2, $fn = 150);
            
        translate([0,0,-11.2])
        cube([17.5,16,10],true);    
            }
            
        rotate([90,0,90])
        translate([0,-23,-4.75])
        cylinder(9.5,hammCylOuterD/2+2,hammCylOuterD/2+2, $fn = 150);    
            
            
        rotate([90,0,90])
        translate([0,-23,-15])
        cylinder(30,8.1/2,8.1/2, $fn = 150);     
        }    
        
        //inner circular parts to touch the bearing    
        difference(){
            rotate([90,0,90])
            translate([0,-23,-8])
            cylinder(16,12.6/2,12.6/2, $fn = 150); 
            
            
            rotate([90,0,90])
            translate([0,-23,-15])
            cylinder(25,8.1/2,8.1/2, $fn = 150);  
            
            rotate([90,0,90])
            translate([0,-23,-4.75+2.3/2])
            cylinder(7.1,hammCylOuterD/2+2,hammCylOuterD/2+2, $fn = 150);  
        }    
            
            
            
        
        
        
    }
}












//Draw a prism based on a 
//right angled triangle
//l - length of prism
//w - width of triangle
//h - height of triangle
module prism(l, w, h) {
       polyhedron(points=[
               [0,0,h],           // 0    front top corner
               [0,0,0],[w,0,0],   // 1, 2 front left & right bottom corners
               [0,l,h],           // 3    back top corner
               [0,l,0],[w,l,0]    // 4, 5 back left & right bottom corners
       ], faces=[ // points for all faces must be ordered clockwise when looking in
               [0,2,1],    // top face
               [3,4,5],    // base face
               [0,1,4,3],  // h face
               [1,2,5,4],  // w face
               [0,3,5,2],  // hypotenuse face
       ]);
}






