%% Accompanying code
% 
% "Measuring Body Vibrations of Stringed Instruments", 187th Meeting of the Acoustical Society of America
% November 21, 2024, 3aMU
%
% Mark Rau
% mrau@mit.edu


close all
clear all
clc

fs = 48000; % sample rate, All recordings were made at a sample rate of 48000 Hz

%% Compare hammer strikes

load('Measurements/mediumHammerImpact.mat')
load('Measurements/smallHammerImpact.mat')
load('Measurements/smallHammerVinylTipImpact.mat')
load('Measurements/smallHammerDoubleHitImpact.mat')


% Time-domain plot
N = length(mediumHammerForce); 
t = (0:N-1)/fs; % time vector

figure(1)
plot(t, mediumHammerForce, 'Linewidth', 2)
hold on
plot(t, smallHammerForce, 'Linewidth', 2)
plot(t, smallHammerVinylTipForce, 'Linewidth', 2)
plot(t, smallHammerDoubleHitForce, 'Linewidth', 2)
grid on
xlim([0,0.006])
xlabel('Time (s)')
ylabel('Force (N)')
legend('medium hammer', 'small hammer metal tip', 'small hammer vinyl tip', 'small hammer double hit')
title('Hammer Tap Time-Domain Comparison')

set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/hammersTimeDomain.png'), '-m10')



% Frequency-domain plot
M = fs;
f = (0:M-1)*fs/M;
MediumHammerForce = 20*log10(abs(fft(mediumHammerForce(1:M))));
SmallHammerForce = 20*log10(abs(fft(smallHammerForce(1:M))));
SmallHammerVinylTipForce = 20*log10(abs(fft(smallHammerVinylTipForce(1:M))));
SmallHammerDoubleHitForce = 20*log10(abs(fft(smallHammerDoubleHitForce(1:M))));


figure(2)
semilogx(f, MediumHammerForce, 'Linewidth', 2)
hold on
semilogx(f, SmallHammerForce, 'Linewidth', 2)
semilogx(f, SmallHammerVinylTipForce, 'Linewidth', 2)
semilogx(f, SmallHammerDoubleHitForce, 'Linewidth', 2)
grid on
xlim([20, 20000])
ylim([-20,60])
xlabel('Frequency (Hz)')
ylabel('Force (dB, N)')
legend('medium hammer', 'small hammer metal tip', 'small hammer vinyl tip', 'small hammer double hit')
title('Hammer Tap Frequency-Domain Comparison')

set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/hammersFrequencyDomain.png'), '-m10')



% Frequency-domain windowed plot
mediumHammerForce_windowed = [mediumHammerForce(1:122); zeros(N-122-1,1)];
smallHammerForce_windowed = [smallHammerForce(1:9); zeros(N-9-1,1)];
smallHammerVinylTipForce_windowed = [smallHammerVinylTipForce(1:30); zeros(N-30-1,1)];
smallHammerDoubleHitForce_windowed = [smallHammerDoubleHitForce(1:245); zeros(N-245-1,1)];


M = fs;
f = (0:M-1)*fs/M;
MediumHammerForce_windowed = 20*log10(abs(fft(mediumHammerForce_windowed(1:M))));
SmallHammerForce_windowed = 20*log10(abs(fft(smallHammerForce_windowed(1:M))));
SmallHammerVinylTipForce_windowed = 20*log10(abs(fft(smallHammerVinylTipForce_windowed(1:M))));
SmallHammerDoubleHitForce_windowed = 20*log10(abs(fft(smallHammerDoubleHitForce_windowed(1:M))));


figure(3)
semilogx(f, MediumHammerForce_windowed, 'Linewidth', 2)
hold on
semilogx(f, SmallHammerForce_windowed, 'Linewidth', 2)
semilogx(f, SmallHammerVinylTipForce_windowed, 'Linewidth', 2)
semilogx(f, SmallHammerDoubleHitForce_windowed, 'Linewidth', 2)
grid on
xlim([20, 20000])
ylim([-20,60])
xlabel('Frequency (Hz)')
ylabel('Force (dB, N)')
legend('medium hammer', 'small hammer metal tip', 'small hammer vinyl tip', 'small hammer double hit')
title('Hammer Tap Frequency-Domain Comparison - Windowed')

set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/hammersFrequencyDomainWindowed.png'), '-m10')



%% Shaker Measurement
load('Measurements/shakerLogSweep_10s_50to20000Hz.mat')


% Time-domain plot
N = length(shakerLogSweep_10s_50to20000Hz); 
t = (0:N-1)/fs; % time vector

sweepRaw = shakerLogSweep_10s_50to20000Hz(:,1); % channel 1: raw sweep file
sweepForce = shakerLogSweep_10s_50to20000Hz(:,2); % channel 2: force sensor
sweepVelocity = shakerLogSweep_10s_50to20000Hz(:,3); % channel 3: laser Doppler vibrometer

figure(4)
subplot(3,1,1)
plot(t, sweepRaw, 'Linewidth', 2)
ylabel('Amplitude')
title('Sweep Time-Domain Comparison')
grid on
subplot(3,1,2)
plot(t, sweepForce, 'Linewidth', 2)
ylabel('Force (N)')
grid on
subplot(3,1,3)
plot(t, sweepVelocity, 'Linewidth', 2)
grid on
xlabel('Time (s)')
ylabel('Velocity (m s^{-1})')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/sweepTimeDomain.png'), '-m10')


% Frequency-domain plot
N = length(shakerLogSweep_10s_50to20000Hz); 
f = (0:N-1)*fs/N; % frequency vector

SweepRaw = 20*log10(abs(fft(shakerLogSweep_10s_50to20000Hz(:,1)))); % channel 1: raw sweep file
SweepForce = 20*log10(abs(fft(shakerLogSweep_10s_50to20000Hz(:,2)))); % channel 2: force sensor
SweepVelocity = 20*log10(abs(fft(shakerLogSweep_10s_50to20000Hz(:,3)))); % channel 3: laser Doppler vibrometer

figure(5)
subplot(3,1,1)
semilogx(f, SweepRaw, 'Linewidth', 2)
ylabel('Magnitude (dB)')
xlim([20,20000])
title('Sweep Frequency-Domain Comparison')
grid on

subplot(3,1,2)
semilogx(f, SweepForce, 'Linewidth', 2)
ylabel('Force (dB, N)')
grid on
xlim([20,20000])

subplot(3,1,3)
semilogx(f, SweepVelocity, 'Linewidth', 2)
grid on
xlim([20,20000])
xlabel('Frequency (Hz)')
ylabel('Velocity (dB, m s^{-1})')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/sweepFrequencyDomain.png'), '-m10')



% Sweep admittance

Admittance = 20*log10(abs(fft(sweepVelocity)./fft(sweepForce)));

figure(6)
semilogx(f, Admittance, 'Linewidth', 2)
xlim([20,20000])
xlabel('Frequency (Hz)')
ylabel('Velocity (dB, s kg^{-1})')
grid on
title('Sweep Admittance')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/sweepAdmittance.png'), '-m10')



%% Accelerometer vs. laser Doppler vibrometer
load("Measurements/admittance_1.mat")
% Channel 1: impact hammer force (N)
% Channel 2: laser Doppler vibrometer velocity (m/s)
% Channel 3: accelerometer acceleration (m/s^2)

velocity = admittance(:,2);
acceleration = admittance(:,3);

Velocity = 20*log10(abs(fft(velocity)));
Acceleration = 20*log10(abs(fft(acceleration)));


% Direct comparison
N = length(velocity); 
t = (0:N-1)/fs; % time vector
f = (0:N-1)*fs/N; % frequency vector

figure(7)
subplot(2,1,1)
plot(t, acceleration, 'Linewidth', 2)
hold on
plot(t, velocity, 'Linewidth', 2)
grid on
xlim([0,0.01])
xlabel('Time (s)')
ylabel('Amplitude')
legend('Accelerometer (m s^{-2})', 'LDV (m s^{-1})')
title('Accelerometer vs. LDV Comparison')

subplot(2,1,2)
semilogx(f, Acceleration, 'Linewidth', 2)
hold on
semilogx(f, Velocity, 'Linewidth', 2)
grid on
xlim([20,20000])
xlabel('Frequency (Hz)')
ylabel('Magnitude (dB0')
% legend('Accelerometer (m s^{-2})', 'LDV (m s^{-1})')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/accelerometerLDV.png'), '-m10')



% Integrated Accelerometer

velocityFromAccelerometer = cumtrapz(t,acceleration); % numerical integration with cumtrapz
VelocityFromAccelerometer = 20*log10(abs(fft(velocityFromAccelerometer)));

figure(8)
subplot(2,1,1)
plot(t, velocityFromAccelerometer, 'Linewidth', 2)
hold on
plot(t, velocity, 'Linewidth', 2)
grid on
xlim([0,0.01])
xlabel('Time (s)')
ylabel('Amplitude')
legend('Integrated Accelerometer (m s^{-1})', 'LDV (m s^{-1})')
title('Integrated Accelerometer vs. LDV Comparison')

subplot(2,1,2)
semilogx(f, VelocityFromAccelerometer, 'Linewidth', 2)
hold on
semilogx(f, Velocity, 'Linewidth', 2)
grid on
xlim([20,20000])
xlabel('Frequency (Hz)')
ylabel('Magnitude (dB0')
legend('Accelerometer (m s^{-2})', 'LDV (m s^{-1})')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/IntegratedAccelerometerLDV.png'), '-m10')




%% Hammer tap LDV admittance
N = fs;
admittance_force = zeros(N,5);
admittance_velocity = zeros(N,5);
admittance_acceleration = zeros(N,5);


for n = 1:5
    % Channel 1: impact hammer force (N)
    % Channel 2: laser Doppler vibrometer velocity (m/s)
    % Channel 3: accelerometer acceleration (m/s^2)
    load(strcat("Measurements/admittance_",num2str(n),".mat"));
    admittance_force(:,n) = admittance(:,1);
    admittance_velocity(:,n) = admittance(:,2);
    admittance_acceleration(:,n) = admittance(:,3);
end

% Window the hammer taps so everything after the strike is zero
admittance_force(6:fs,:) = zeros(fs-5,5);


% Calculate the admittance for each measurement

Admittance = fft(admittance_velocity)./fft(admittance_force);
f = (0:N-1)*fs/N;

figure(9)
subplot(3,1,1)
semilogx(f, 20*log10(abs(fft(admittance_force))),'Linewidth', 2)
ylabel('Force (dB, N)')
grid on
xlim([20,20000])
ylim([-20,40])
title('Admittance')

subplot(3,1,2)
semilogx(f, 20*log10(abs(fft(admittance_velocity))),'Linewidth', 2)
ylabel('Velocity (dB, m s^{-1})')
grid on
xlim([20,20000])
ylim([-80,20])

subplot(3,1,3)
semilogx(f, 20*log10(abs(Admittance)), 'Linewidth', 2)
ylabel('Admittance (dB, s kg^{-1})')
grid on
xlim([20,20000])
ylim([-100,40])

xlabel('Frequency (Hz)')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/admittance.png'), '-m10')


% Show the coherence
[frf,f_coh,coh] = modalfrf(admittance_force(:), admittance_velocity(:), fs, fs,'Sensor','dis');


figure(10)
semilogx(f, 20*log10(abs(Admittance)), 'Linewidth', 2)
ylabel('Admittance (dB, s kg^{-1})')
yyaxis right
semilogx(f_coh, coh, 'Linewidth', 2)
xlim([20,20000])
xlabel('Frequency (Hz)')
ylabel('Coherence')
grid on
title('Coherence')

set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/Coherence.png'), '-m10')



%% Averaged Admittance
meanAdmittanceDB = mean(20*log10(abs(Admittance)),2);
stdAdmittanceDB = std(20*log10(abs(Admittance))')';

lowerLim = meanAdmittanceDB-stdAdmittanceDB;
upperLim = meanAdmittanceDB+stdAdmittanceDB;

meanAdmittancePhase = mean(angle(Admittance),2);
stdAdmittancePhase = std(angle(Admittance)')';

lowerLimPhase = meanAdmittancePhase-stdAdmittancePhase;
upperLimPhase = meanAdmittancePhase+stdAdmittancePhase;



figure(11)
subplot(3,1,1:2)
plot(f(2:end), lowerLim(2:end) , 'k', 'LineWidth', 1.5);
hold on;
plot(f(2:end), upperLim(2:end) , 'k', 'LineWidth', 1.5);
patch([f(2:end) fliplr(f(2:end))], [lowerLim(2:end)' fliplr(upperLim(2:end)')], 'k')

plot(f, meanAdmittanceDB, 'r', 'Linewidth', 1.5)
ylabel('Admittance (dB, s kg^{-1})')
grid on
set(gca,'Xscale','log')
xlim([20,20000])
ylim([-80,0])
title('Averaged Admittance')
legend('','','\pm 1 std','mean')

subplot(3,1,3)
plot(f(2:end), lowerLimPhase(2:end) , 'k', 'LineWidth', 1.5);
hold on;
plot(f(2:end), upperLimPhase(2:end) , 'k', 'LineWidth', 1.5);
patch([f(2:end) fliplr(f(2:end))], [lowerLimPhase(2:end)' fliplr(upperLimPhase(2:end)')], 'k')

plot(f, meanAdmittancePhase, 'r', 'Linewidth', 1.5)
ylabel('Phase (rad)')
grid on
set(gca,'Xscale','log')
ylim([-2,2])
xlim([20,20000])
xlabel('Frequency (Hz)')


set(gcf, 'Color', 'w');
set(gcf, 'Position', [100 100 1000 400]);
% export_fig(strcat('Figures/AveragedAdmittance.png'), '-m10')

























